/*
 * zlib.c - zlib support for Ruby
 *
 *   Copyright (C) Ueno Katsuhiro 2000
 *
 * $Id: zlib.c,v 1.15 2000/07/15 09:24:46 katsu Exp $
 */

#include <time.h>
#include <zlib.h>
#include "ruby.h"

#ifndef GZIP_SUPPORT
#define GZIP_SUPPORT  1
#endif


/* from zutil.h */
#ifndef DEF_MEM_LEVEL
#if MAX_MEM_LEVEL >= 8
#define DEF_MEM_LEVEL  8
#else
#define DEF_MEM_LEVEL  MAX_MEM_LEVEL
#endif
#endif



static VALUE cZError, cStreamEnd, cNeedDict;
static VALUE cStreamError, cDataError, cMemError, cBufError, cVersionError;



static void raise_zlib_error(int err, const char *msg) NORETURN;

static void raise_zlib_error(int err, const char *msg)
{
	if (!msg)
		msg = zError(err);

	switch(err) {
	  case Z_STREAM_END:
		rb_raise(cStreamEnd, "%s", msg);
	  case Z_NEED_DICT:
		rb_raise(cNeedDict, "%s", msg);
	  case Z_STREAM_ERROR:
		rb_raise(cStreamError, "%s", msg);
	  case Z_DATA_ERROR:
		rb_raise(cDataError, "%s", msg);
	  case Z_BUF_ERROR:
		rb_raise(cBufError, "%s", msg);
	  case Z_VERSION_ERROR:
		rb_raise(cVersionError, "%s", msg);
	  case Z_MEM_ERROR:
		rb_raise(cMemError, "%s", msg);
	  case Z_ERRNO:
		rb_sys_fail(msg);
	  default:
		rb_raise(cZError, "unknown zlib error %d: %s", err, msg);
	}
}



VALUE rb_zlib_version(VALUE klass)
{
	return rb_str_new2(zlibVersion());
}



static VALUE do_checksum(int argc, VALUE *argv,
			 uLong (*func)(uLong, const Bytef *, uInt))
{
	VALUE str, vsum;
	const char *buf;
	unsigned long sum;
	int len;

	rb_scan_args(argc, argv, "02", &str, &vsum);

	if (!NIL_P(str)) {
		buf = str2cstr(str, &len);
	} else {
		buf = Z_NULL;
		len = 0;
	}
	if (!NIL_P(vsum))
		sum = NUM2ULONG(vsum);
	else
		sum = func(0, Z_NULL, 0);

	sum = func(sum, buf, len);
	return rb_uint2inum(sum);
}



VALUE rb_zlib_adler32(int argc, VALUE *argv, VALUE klass)
{
	return do_checksum(argc, argv, adler32);
}



VALUE rb_zlib_crc32(int argc, VALUE *argv, VALUE klass)
{
	return do_checksum(argc, argv, crc32);
}



VALUE rb_zlib_crc_table(VALUE obj)
{
	const unsigned long *crctbl;
	VALUE dst;
	int i;

	crctbl = get_crc_table();
	dst = rb_ary_new2(256);

	for (i = 0; i < 256; i++)
		rb_ary_push(dst, rb_uint2inum(crctbl[i]));
	return dst;
}



struct zstream {
	unsigned long flag;
	VALUE buf;
	long buf_filled;
	z_stream stream;
	int (*reset)(z_streamp);
	int (*end)(z_streamp);
	int (*run)(z_streamp, int);
};


#define ZSTREAM_FLAG_READY      0x1
#define ZSTREAM_FLAG_IN_STREAM  0x2
#define ZSTREAM_FLAG_FINISHED   0x4
#define ZSTREAM_FLAG_SPECIAL    0x10

/* I think that more better value should be found,
   but I gave up finding it. B) */
#define ZSTREAM_INITIAL_BUFSIZE  1024
#define ZSTREAM_AVAIL_OUT_STEP   65536
#define ZSTREAM_AVAIL_OUT_MIN    1024


#define ZSTREAM_INIT(z, resetfunc, endfunc, dofunc) do { \
	(z)->flag = 0; \
	(z)->buf = Qnil; \
	(z)->buf_filled = 0; \
	(z)->stream.zalloc = Z_NULL; \
	(z)->stream.zfree = Z_NULL; \
	(z)->stream.opaque = Z_NULL; \
	(z)->stream.msg = Z_NULL; \
	(z)->stream.next_out = Z_NULL; \
	(z)->stream.avail_out = 0; \
	(z)->reset = (resetfunc); \
	(z)->end = (endfunc); \
	(z)->run = (dofunc); \
} while (0)


#define ZSTREAM_READY(z)  ((z)->flag |= ZSTREAM_FLAG_READY)



static struct zstream *get_zstream(VALUE obj)
{
	struct zstream *z;

	Data_Get_Struct(obj, struct zstream, z);
	if (!(z->flag & ZSTREAM_FLAG_READY))
		rb_raise(cZError, "stream is not ready");

	return z;
}



static void zstream_expand_buffer(struct zstream *z)
{
	long inc;

	if (NIL_P(z->buf)) {
		z->buf = rb_str_new(0, ZSTREAM_INITIAL_BUFSIZE);
		z->buf_filled = 0;
		z->stream.next_out = RSTRING(z->buf)->ptr;
		z->stream.avail_out = ZSTREAM_INITIAL_BUFSIZE;
		return;
	}

	if (RSTRING(z->buf)->len - z->buf_filled >= ZSTREAM_AVAIL_OUT_STEP) {
		/* to keep other threads from freezing */
		z->stream.avail_out = ZSTREAM_AVAIL_OUT_STEP;
	} else {
		inc = z->buf_filled / 2;
		if (inc < ZSTREAM_AVAIL_OUT_MIN)
			inc = ZSTREAM_AVAIL_OUT_MIN;
		rb_str_resize(z->buf, z->buf_filled + inc);
		z->stream.avail_out = (inc < ZSTREAM_AVAIL_OUT_STEP) ?
			inc : ZSTREAM_AVAIL_OUT_STEP;
	}
	z->stream.next_out = RSTRING(z->buf)->ptr + z->buf_filled;
}



static void zstream_append_buffer(struct zstream *z, const char *src, int len)
{
	if (NIL_P(z->buf)) {
		z->buf = rb_str_new(src, len);
		z->buf_filled = len;
		z->stream.next_out = RSTRING(z->buf)->ptr;
		z->stream.avail_out = 0;
		return;
	}

	if (RSTRING(z->buf)->len < z->buf_filled + len) {
		rb_str_resize(z->buf, z->buf_filled + len);
		z->stream.avail_out = 0;
	} else {
		if (z->stream.avail_out >= len)
			z->stream.avail_out -= len;
		else
			z->stream.avail_out = 0;
	}
	memcpy(RSTRING(z->buf)->ptr + z->buf_filled, src, len);
	z->buf_filled += len;
	z->stream.next_out = RSTRING(z->buf)->ptr + z->buf_filled;
}



static VALUE zstream_detach_buffer(struct zstream *z)
{
	VALUE dst;

	if (NIL_P(z->buf)) {
		dst = rb_str_new(0, 0);
	} else {
		dst = z->buf;
		rb_str_resize(dst, z->buf_filled);
	}

	z->buf = Qnil;
	z->buf_filled = 0;
	z->stream.next_out = 0;
	z->stream.avail_out = 0;
	return dst;
}


#if GZIP_SUPPORT

static VALUE zstream_shift_buffer(struct zstream *z, int len)
{
	VALUE dst;

	if (z->buf_filled <= len)
		return zstream_detach_buffer(z);

	dst = rb_str_substr(z->buf, 0, len);
	z->buf_filled -= len;
	memmove(RSTRING(z->buf)->ptr, RSTRING(z->buf)->ptr + len,
		z->buf_filled);
	z->stream.next_out = RSTRING(z->buf)->ptr + z->buf_filled;
	z->stream.avail_out = (z->buf_filled < ZSTREAM_AVAIL_OUT_STEP) ?
		z->buf_filled : ZSTREAM_AVAIL_OUT_STEP;

	return dst;
}



static void zstream_buffer_ungetc(struct zstream *z, char c)
{
	if (NIL_P(z->buf) || RSTRING(z->buf)->len - z->buf_filled == 0)
		zstream_expand_buffer(z);

	memmove(RSTRING(z->buf)->ptr + 1, RSTRING(z->buf)->ptr, z->buf_filled);
	RSTRING(z->buf)->ptr[0] = c;
	z->buf_filled++;
	if (z->stream.avail_out > 0) {
		z->stream.next_out++;
		z->stream.avail_out--;
	}
}

#endif



static void zstream_reset(struct zstream *z)
{
	int err;

	err = z->reset(&z->stream);
	if (err != Z_OK)
		raise_zlib_error(err, z->stream.msg);
	z->flag = ZSTREAM_FLAG_READY;
	z->buf = Qnil;
	z->buf_filled = 0;
	z->stream.next_out = 0;
	z->stream.avail_out = 0;
}



static void zstream_end(struct zstream *z)
{
	int err;

	if (!(z->flag & ZSTREAM_FLAG_READY)) {
		if (RTEST(ruby_debug))
			rb_warning("zstream_end() is called"
				   " but the stream is not ready.");
		return;
	}
	if (z->flag & ZSTREAM_FLAG_IN_STREAM) {
		if (RTEST(ruby_debug))
			rb_warning("zstream_end() is called"
				   " but the stream is not finished.");
		zstream_reset(z);
	}

	z->flag = 0;
	z->buf = Qnil;
	z->buf_filled = 0;
	z->stream.next_out = 0;
	z->stream.avail_out = 0;

	err = z->end(&z->stream);
	if (err != Z_OK)
		raise_zlib_error(err, z->stream.msg);
}



void zstream_mark(struct zstream *z)
{
	rb_gc_mark(z->buf);
}



void zstream_free(struct zstream *z)
{
	zstream_end(z);
	free(z);
}



VALUE rb_zstream_end(VALUE obj)
{
	zstream_end(get_zstream(obj));
	return Qnil;
}



VALUE rb_zstream_reset(VALUE obj)
{
	zstream_reset(get_zstream(obj));
	return Qnil;
}



static void zstream_run_internal(struct zstream *z, int flush)
{
	int err;
	unsigned long start;

	if (flush != Z_FINISH && z->stream.avail_in == 0)
		return;

	do {
		if (z->stream.avail_out == 0)
			zstream_expand_buffer(z);

		rb_thread_schedule();
		start = z->stream.total_out;
		err = z->run(&z->stream, flush);
		z->buf_filled += z->stream.total_out - start;

		if (err == Z_STREAM_END)
			break;
		if (err != Z_OK)
			raise_zlib_error(err, z->stream.msg);
	} while (flush == Z_FINISH || z->stream.avail_in > 0);

	if (err != Z_STREAM_END)
		z->flag |= ZSTREAM_FLAG_IN_STREAM;
	else {
		z->flag &= ~ZSTREAM_FLAG_IN_STREAM;
		z->flag |= ZSTREAM_FLAG_FINISHED;
	}
}



static void zstream_run(struct zstream *z, VALUE src, int flush)
{
	int n;

	if (NIL_P(src)) {
		flush = Z_FINISH;
		z->stream.avail_in = 0;
	} else {
		z->stream.next_in = str2cstr(src, &n);
		/* the type of z->stream.avail_in may not be int. */
		z->stream.avail_in = n;
	}
	zstream_run_internal(z, flush);
}



VALUE rb_zstream_finish(VALUE obj)
{
	struct zstream *z = get_zstream(obj);
	VALUE dst;

	zstream_run(z, Qnil, Z_FINISH);
	dst = zstream_detach_buffer(z);

	if (OBJ_TAINTED(obj))
		rb_obj_taint(dst);
	return dst;
}



VALUE rb_zstream_flush_out(VALUE obj)
{
	struct zstream *z = get_zstream(obj);
	VALUE dst;

	dst = zstream_detach_buffer(z);
	if (OBJ_TAINTED(obj))
		rb_obj_taint(dst);
	return dst;
}



VALUE rb_zstream_total_in(VALUE obj)
{
	return rb_uint2inum(get_zstream(obj)->stream.total_in);
}

VALUE rb_zstream_total_out(VALUE obj)
{
	return rb_uint2inum(get_zstream(obj)->stream.total_out);
}

VALUE rb_zstream_data_type(VALUE obj)
{
	return INT2FIX(get_zstream(obj)->stream.data_type);
}

VALUE rb_zstream_adler(VALUE obj)
{
	return rb_uint2inum(get_zstream(obj)->stream.adler);
}

VALUE rb_zstream_finished_p(VALUE obj)
{
	return (get_zstream(obj)->flag & ZSTREAM_FLAG_FINISHED)
		? Qtrue : Qfalse;
}

VALUE rb_zstream_closed_p(VALUE obj)
{
	struct zstream *z;
	Data_Get_Struct(obj, struct zstream, z);
	return (z->flag & ZSTREAM_FLAG_READY) ? Qfalse : Qtrue;
}



static int value_to_compression_level(VALUE val)
{
	int level;

	if (NIL_P(val))
		return Z_DEFAULT_COMPRESSION;

	Check_Type(val, T_FIXNUM);
	level = FIX2INT(val);
	if (!(level == Z_DEFAULT_COMPRESSION || (0 <= level && level <= 9)))
		rb_raise(rb_eArgError, "invalid compression level");
	return level;
}



static int value_to_window_bits(VALUE val)
{
	int wbits;

	if (NIL_P(val))
		return MAX_WBITS;

	Check_Type(val, T_FIXNUM);
	wbits = FIX2INT(val);
	if (wbits < 8 || MAX_WBITS < wbits)
		rb_raise(rb_eArgError, "window bits out of range");
	return wbits;
}



static int value_to_memlevel(VALUE val)
{
	int memlevel;

	if (NIL_P(val))
		return DEF_MEM_LEVEL;

	Check_Type(val, T_FIXNUM);
	memlevel = FIX2INT(val);
	if (memlevel < 1 || MAX_MEM_LEVEL < memlevel)
		rb_raise(rb_eArgError, "memory level out of range");
	return memlevel;
}



static int value_to_strategy(VALUE val)
{
	int strategy;

	if (NIL_P(val))
		return Z_DEFAULT_STRATEGY;

	Check_Type(val, T_FIXNUM);
	strategy = FIX2INT(val);
	if (strategy != Z_DEFAULT_STRATEGY
	    && strategy != Z_FILTERED
	    && strategy != Z_HUFFMAN_ONLY)
		rb_raise(rb_eArgError, "invalid strategy");
	return strategy;
}



static int value_to_flush(VALUE val)
{
	int flush;

	if (NIL_P(val))
		return Z_NO_FLUSH;

	Check_Type(val, T_FIXNUM);
	flush = FIX2INT(val);
	if (flush != Z_NO_FLUSH
	    && flush != Z_SYNC_FLUSH
	    && flush != Z_FULL_FLUSH)
		rb_raise(rb_eArgError, "invalid flush method");
	return flush;
}



#define DATA_MAKE_ZSTREAM(klass, z) \
	Data_Make_Struct((klass), struct zstream, \
			 zstream_mark, zstream_free, (z))


#define ZSTREAM_INIT_DEFLATE(z) \
	ZSTREAM_INIT(z, deflateReset, deflateEnd, deflate)



static void deflate_init(struct zstream *z, int level)
{
	int err;

	ZSTREAM_INIT_DEFLATE(z);

	err = deflateInit(&z->stream, level);
	if (err != Z_OK)
		raise_zlib_error(err, z->stream.msg);

	ZSTREAM_READY(z);
}



VALUE rb_deflate_s_deflate(int argc, VALUE *argv, VALUE klass)
{
	VALUE src, v_level, dst;
	int level;
	struct zstream z;

	rb_scan_args(argc, argv, "11", &src, &v_level);
	level = value_to_compression_level(v_level);

	deflate_init(&z, level);
	zstream_run(&z, src, Z_FINISH);
	dst = zstream_detach_buffer(&z);
	zstream_end(&z);

 	if (OBJ_TAINTED(src))
		rb_obj_taint(dst);
	return dst;
}



VALUE rb_deflate_s_new(int argc, VALUE *argv, VALUE klass)
{
	VALUE v_level, v_wbits, v_memlevel, v_strategy, dst;
	int level, wbits, memlevel, strategy;
	int err;
	struct zstream *z;

	rb_scan_args(argc, argv, "04",
		     &v_level, &v_wbits, &v_memlevel, &v_strategy);
	level = value_to_compression_level(v_level);

	dst = DATA_MAKE_ZSTREAM(klass, z);

	if (argc <= 1) {
		deflate_init(z, level);
		return dst;
	}

	wbits = value_to_window_bits(v_wbits);
	memlevel = value_to_memlevel(v_memlevel);
	strategy = value_to_strategy(v_strategy);

	ZSTREAM_INIT_DEFLATE(z);
	err = deflateInit2(&z->stream, level,
			   Z_DEFLATED, wbits, memlevel, strategy);
	if (err != Z_OK)
		raise_zlib_error(err, z->stream.msg);
	ZSTREAM_READY(z);

	return dst;
}



VALUE rb_deflate_clone(VALUE obj)
{
	struct zstream *z = get_zstream(obj);
	struct zstream *z2;
	VALUE dst;
	int err;

	dst = DATA_MAKE_ZSTREAM(rb_class_of(obj), z2);

	ZSTREAM_INIT_DEFLATE(z2);
	err = deflateCopy(&z2->stream, &z->stream);
	if (err != Z_OK)
		raise_zlib_error(err, 0);
	z2->flag = z->flag;

	if (OBJ_TAINTED(obj))
		rb_obj_taint(dst);
	return dst;
}



VALUE rb_deflate_deflate(int argc, VALUE *argv, VALUE obj)
{
	struct zstream *z = get_zstream(obj);
	VALUE src, v_flush, dst;
	int flush;

	rb_scan_args(argc, argv, "11", &src, &v_flush);

	flush = value_to_flush(v_flush);
	zstream_run(z, src, flush);
	dst = zstream_detach_buffer(z);

	if (OBJ_TAINTED(obj) || OBJ_TAINTED(src)) {
		rb_obj_taint(obj);
		rb_obj_taint(dst);
	}
	return dst;
}



VALUE rb_deflate_addstr(VALUE obj, VALUE src)
{
	zstream_run(get_zstream(obj), src, Z_NO_FLUSH);
	if (OBJ_TAINTED(src))
		rb_obj_taint(obj);
	return obj;
}



VALUE rb_deflate_params(VALUE obj, VALUE v_level, VALUE v_strategy)
{
	struct zstream *z = get_zstream(obj);
	int level, strategy;
	int err;

	level = value_to_compression_level(v_level);
	strategy = value_to_strategy(v_strategy);

	err = deflateParams(&z->stream, level, strategy);
	while (err == Z_BUF_ERROR) {
		rb_warning("deflateParams() returned Z_BUF_ERROR");
		zstream_expand_buffer(z);
		err = deflateParams(&z->stream, level, strategy);
	}
	if (err != Z_OK)
		raise_zlib_error(err, z->stream.msg);

	return Qnil;
}



VALUE rb_deflate_set_dictionary(VALUE obj, VALUE v_dic)
{
	struct zstream *z = get_zstream(obj);
	int diclen;
	char *dic;
	int err;

	dic = str2cstr(v_dic, &diclen);
	err = deflateSetDictionary(&z->stream, dic, diclen);
	if (err != Z_OK)
		raise_zlib_error(err, z->stream.msg);

	if (OBJ_TAINTED(v_dic))
		rb_obj_taint(obj);
	return v_dic;
}



#define ZSTREAM_INIT_INFLATE(z) \
	ZSTREAM_INIT(z, inflateReset, inflateEnd, inflate)

#define INFLATE_FLAG_SYNC   ZSTREAM_FLAG_SPECIAL   /* out of use now */



static void inflate_init(struct zstream *z)
{
	int err;

	ZSTREAM_INIT_INFLATE(z);

	err = inflateInit(&z->stream);
	if (err != Z_OK)
		raise_zlib_error(err, z->stream.msg);

	ZSTREAM_READY(z);
}



VALUE rb_inflate_s_inflate(VALUE obj, VALUE src)
{
	VALUE dst;
	struct zstream z;

	inflate_init(&z);
	zstream_run(&z, src, Z_SYNC_FLUSH);
	zstream_run(&z, Qnil, Z_FINISH);
	dst = zstream_detach_buffer(&z);
	zstream_end(&z);

 	if (OBJ_TAINTED(src))
		rb_obj_taint(dst);
	return dst;
}



VALUE rb_inflate_s_new(int argc, VALUE *argv, VALUE klass)
{
	VALUE v_wbits, dst;
	int wbits;
	int err;
	struct zstream *z;

	rb_scan_args(argc, argv, "01", &v_wbits);

	dst = DATA_MAKE_ZSTREAM(klass, z);

	if (argc == 0) {
		inflate_init(z);
		return dst;
	}

	wbits = value_to_window_bits(v_wbits);

	ZSTREAM_INIT_INFLATE(z);
	err = inflateInit2(&z->stream, wbits);
	if (err != Z_OK)
		raise_zlib_error(err, z->stream.msg);
	ZSTREAM_READY(z);

	return dst;
}



VALUE rb_inflate_inflate(VALUE obj, VALUE src)
{
	struct zstream *z = get_zstream(obj);
	VALUE dst;
	int len;
	char *p;

	if (z->flag & ZSTREAM_FLAG_FINISHED) {
		if (NIL_P(src))
			dst = zstream_detach_buffer(z);
		else {
			p = str2cstr(src, &len);
			zstream_append_buffer(z, p, len);
			dst = rb_str_new(0, 0);
		}
	} else {
		zstream_run(z, src, Z_SYNC_FLUSH);
		dst = zstream_detach_buffer(z);

		if (z->flag & ZSTREAM_FLAG_FINISHED)
			zstream_append_buffer(z, z->stream.next_in,
					      z->stream.avail_in);
	}

	if (OBJ_TAINTED(obj) || OBJ_TAINTED(src)) {
		rb_obj_taint(obj);
		rb_obj_taint(dst);
	}
	return dst;
}



VALUE rb_inflate_addstr(VALUE obj, VALUE src)
{
	struct zstream *z = get_zstream(obj);
	int len;
	char *p;

	if (z->flag & ZSTREAM_FLAG_FINISHED) {
		if (!NIL_P(src)) {
			p = str2cstr(src, &len);
			zstream_append_buffer(z, p, len);
		}
	} else {
		zstream_run(z, src, Z_SYNC_FLUSH);
		if (z->flag & ZSTREAM_FLAG_FINISHED)
			zstream_append_buffer(z, z->stream.next_in,
					      z->stream.avail_in);
	}

	if (OBJ_TAINTED(src))
		rb_obj_taint(obj);
	return obj;
}



#if 0
VALUE rb_inflate_sync(VALUE obj)
{
	struct zstream *z = get_zstream(obj);
	int err;

	err = inflateSync(&z->stream);
	if (err != Z_OK)
		raise_zlib_error(err, z->stream.msg);
	return Qnil;
}
#endif



VALUE rb_inflate_sync_point_p(VALUE obj)
{
	struct zstream *z = get_zstream(obj);
	int err;

	err = inflateSyncPoint(&z->stream);
	if (err == 1)
		return Qtrue;
	if (err != Z_OK)
		raise_zlib_error(err, z->stream.msg);
	return Qfalse;
}



VALUE rb_inflate_set_dictionary(VALUE obj, VALUE v_dic)
{
	struct zstream *z = get_zstream(obj);
	int diclen;
	char *dic;
	int err;

	dic = str2cstr(v_dic, &diclen);
	err = inflateSetDictionary(&z->stream, dic, diclen);
	if (err != Z_OK)
		raise_zlib_error(err, z->stream.msg);

	if (OBJ_TAINTED(v_dic))
		rb_obj_taint(obj);
	return v_dic;
}




#if GZIP_SUPPORT

/*
 * WARNING:
 *   these gzip* functions are using undocumented feature of zlib.
 *   I don't use gzFile functions because I don't think that
 *   these functions are good for Ruby.
 *   I also don't feel that they are useful :-p
 */

/* .gz file header */
#define GZ_MAGIC1  0x1f
#define GZ_MAGIC2  0x8b
#define GZ_METHOD_DEFLATE  8
#define GZ_FLAG_MULTIPART     0x2
#define GZ_FLAG_EXTRA         0x4
#define GZ_FLAG_ORIG_NAME     0x8
#define GZ_FLAG_COMMENT       0x10
#define GZ_FLAG_ENCRYPT       0x20
#define GZ_FLAG_UNKNOWN_MASK  0xc0

#define GZ_EXTRAFLAG_FAST  0x4
#define GZ_EXTRAFLAG_SLOW  0x2


#define OS_MSDOS    0x00
#define OS_AMIGA    0x01
#define OS_VMS      0x02
#define OS_UNIX     0x03
#define OS_ATARI    0x05
#define OS_OS2      0x06
#define OS_MACOS    0x07
#define OS_TOPS20   0x0a
#define OS_WIN32    0x0b

#ifndef OS_CODE
#define OS_CODE  OS_UNIX
#endif



static ID id_write;
static ID id_read;
static ID id_flush;
static ID id_seek;
static ID id_close;
static VALUE cGzError, cNoFooter, cCRCError, cLengthError;



struct gzfile {
	struct zstream z;
	VALUE io;
	unsigned long crc;
	time_t mtime;
	int level;       /* for GzipWriter */
	int os_code;
	VALUE orig_name;
	VALUE comment;
	VALUE unused;    /* for GzipReader */
	int lineno;
	int ungetc;
	void (*close)(struct gzfile *);
};


/* for GzipWriter */
#define GZIP_FLAG_SYNC             ZSTREAM_FLAG_SPECIAL
#define GZIP_FLAG_HEADER_FINISHED  (ZSTREAM_FLAG_SPECIAL << 1)
/* for GzipReader */
#define GZIP_FLAG_FOOTER_FINISHED  (ZSTREAM_FLAG_SPECIAL << 2)

#define GZIP_READ_SIZE     2048



#define GZFILE_INIT(gz, type, io_obj, close_func) do { \
	ZSTREAM_INIT_##type(&((gz)->z)); \
	(gz)->io = (io_obj); \
	(gz)->crc = crc32(0, Z_NULL, 0); \
	(gz)->mtime = 0; \
	(gz)->level = 0; \
	(gz)->os_code = OS_CODE; \
	(gz)->orig_name = Qnil; \
	(gz)->comment = Qnil; \
	(gz)->unused = Qnil; \
	(gz)->lineno = 0; \
	(gz)->ungetc = 0; \
	(gz)->close = (close_func); \
} while (0)


#define GZFILE_ZSTREAM_FINISHED(gz)  (((gz)->z.flag & ZSTREAM_FLAG_FINISHED) \
				      && (gz)->z.buf_filled == 0)

#define GZFILE_EOF(gz)  (GZFILE_ZSTREAM_FINISHED(gz) \
			 && ((gz)->z.flag & GZIP_FLAG_FOOTER_FINISHED))



static struct gzfile *get_gzfile(VALUE obj)
{
	struct gzfile *gz;

	Data_Get_Struct(obj, struct gzfile, gz);
	if (NIL_P(gz->io))
		rb_raise(cGzError, "closed gzip stream");
	if (!(gz->z.flag & ZSTREAM_FLAG_READY))
		rb_raise(cZError, "stream is not ready");

	return gz;
}



void gzip_mark(struct gzfile *gz)
{
	rb_gc_mark(gz->io);
	rb_gc_mark(gz->orig_name);
	rb_gc_mark(gz->comment);
	rb_gc_mark(gz->unused);
	zstream_mark(&gz->z);
}



void gzip_free(struct gzfile *gz)
{
	if (!NIL_P(gz->io)) {
		/* prevent checking footer */
		gz->z.flag |= GZIP_FLAG_FOOTER_FINISHED;
		gz->close(gz);
	}
	free(gz);
}


#define DATA_MAKE_GZFILE(klass, gz) \
	Data_Make_Struct((klass), struct gzfile, gzip_mark, gzip_free, (gz))



VALUE rb_gzip_to_io(VALUE obj)
{
	return get_gzfile(obj)->io;
}

VALUE rb_gzip_crc(VALUE obj)
{
	return rb_uint2inum(get_gzfile(obj)->crc);
}

VALUE rb_gzip_mtime(VALUE obj)
{
	return rb_time_new(get_gzfile(obj)->mtime, (time_t)0);
}

VALUE rb_gzip_level(VALUE obj)
{
	return INT2FIX(get_gzfile(obj)->level);
}

VALUE rb_gzip_os_code(VALUE obj)
{
	return INT2FIX(get_gzfile(obj)->os_code);
}

VALUE rb_gzip_orig_name(VALUE obj)
{
	return rb_str_dup(get_gzfile(obj)->orig_name);
}

VALUE rb_gzip_comment(VALUE obj)
{
	return rb_str_dup(get_gzfile(obj)->comment);
}

VALUE rb_gzip_lineno(VALUE obj)
{
	return INT2NUM(get_gzfile(obj)->lineno);
}

VALUE rb_gzip_set_lineno(VALUE obj, VALUE lineno)
{
	struct gzfile *gz = get_gzfile(obj);
	gz->lineno = NUM2INT(lineno);
	return lineno;
}



VALUE rb_gzip_set_mtime(VALUE obj, VALUE time)
{
	struct gzfile *gz = get_gzfile(obj);
	VALUE val;

	if (gz->z.flag & GZIP_FLAG_HEADER_FINISHED)
		rb_raise(cGzError, "header is already written");

	if (FIXNUM_P(time))
		gz->mtime = FIX2INT(time);
	else {
		val = rb_Integer(time);
		if (FIXNUM_P(val))
			gz->mtime = FIX2INT(val);
		else
			gz->mtime = rb_big2ulong(val);
	}
	return time;
}



VALUE rb_gzip_set_orig_name(VALUE obj, VALUE str)
{
	struct gzfile *gz = get_gzfile(obj);
	VALUE s;
	char *p;

	if (gz->z.flag & GZIP_FLAG_HEADER_FINISHED)
		rb_raise(cGzError, "header is already written");
	s = rb_str_dup(rb_str_to_str(str));
	p = memchr(RSTRING(s)->ptr, 0, RSTRING(s)->len);
	if (p)
		rb_str_resize(s, p - RSTRING(s)->ptr);
	gz->orig_name = s;
	return str;
}



VALUE rb_gzip_set_comment(VALUE obj, VALUE str)
{
	struct gzfile *gz = get_gzfile(obj);
	VALUE s;
	char *p;

	if (gz->z.flag & GZIP_FLAG_HEADER_FINISHED)
		rb_raise(cGzError, "header is already written");
	s = rb_str_dup(rb_str_to_str(str));
	p = memchr(RSTRING(s)->ptr, 0, RSTRING(s)->len);
	if (p)
		rb_str_resize(s, p - RSTRING(s)->ptr);
	gz->comment = s;
	return str;
}



static VALUE gzip_close(struct gzfile *gz, int closeflag)
{
	VALUE io;
	io = gz->io;
	gz->close(gz);
	if (closeflag && rb_respond_to(io, id_close))
		rb_funcall(io, id_close, 0);
	return io;
}



VALUE rb_gzip_close(int argc, VALUE *argv, VALUE obj)
{
	struct gzfile *gz = get_gzfile(obj);
	VALUE closeflag;
	rb_scan_args(argc, argv, "01", &closeflag);
	return gzip_close(gz, !RTEST(closeflag));
}



static VALUE gzip_safe_close(VALUE obj)
{
	struct gzfile *gz;
	Data_Get_Struct(obj, struct gzfile, gz);
	if (!NIL_P(gz->io))
		gzip_close(gz, 1);
	return Qnil;
}



VALUE rb_gzip_closed_p(VALUE obj)
{
	struct gzfile *gz;
	Data_Get_Struct(obj, struct gzfile, gz);
	return NIL_P(gz->io) ? Qtrue : Qfalse;
}



VALUE rb_gzip_eof_p(VALUE obj)
{
	struct gzfile *gz = get_gzfile(obj);
	return GZFILE_EOF(gz) ? Qtrue : Qfalse;
}



static unsigned int gzip_get16(const unsigned char *src)
{
	unsigned int n;
	n  = *(src++) & 0xff;
	n |= (*(src++) & 0xff) << 8;
	return n;
}



static unsigned long gzip_get32(const unsigned char *src)
{
	unsigned long n;
	n  = *(src++) & 0xff;
	n |= (*(src++) & 0xff) << 8;
	n |= (*(src++) & 0xff) << 16;
	n |= (*(src++) & 0xff) << 24;
	return n;
}



static void gzip_set32(unsigned long n, unsigned char *dst)
{
	*(dst++) = n & 0xff;
	*(dst++) = (n >> 8) & 0xff;
	*(dst++) = (n >> 16) & 0xff;
	*dst     = (n >> 24) & 0xff;
}



static void gzip_write_io(struct gzfile *gz, VALUE str)
{
	OBJ_TAINT(str);
	rb_funcall(gz->io, id_write, 1, str);
	if ((gz->z.flag & GZIP_FLAG_SYNC) && rb_respond_to(gz->io, id_flush))
		rb_funcall(gz->io, id_flush, 0);
}



static VALUE gzip_read_io(struct gzfile *gz, int len)
{
	VALUE str;
	str = rb_funcall(gz->io, id_read, 1, INT2FIX(len));
	if (!NIL_P(str))
		Check_Type(str, T_STRING);
	return str;
}



static void gzipreader_inflate(struct gzfile *gz, VALUE str)
{
	if (NIL_P(str))
		rb_raise(cGzError, "unexpected end of file");
	zstream_run(&gz->z, str, Z_SYNC_FLUSH);
	if (gz->z.flag & ZSTREAM_FLAG_FINISHED)
		gz->unused = rb_str_new(gz->z.stream.next_in,
					gz->z.stream.avail_in);
}



static void gzip_make_header(struct gzfile *gz)
{
	unsigned char buf[10];  /* the size of .gz header */
	unsigned char flags = 0, extraflags = 0;

	if (!NIL_P(gz->orig_name))
		flags |= GZ_FLAG_ORIG_NAME;
	if (!NIL_P(gz->comment))
		flags |= GZ_FLAG_COMMENT;

	if (gz->mtime == 0)
		gz->mtime = time(0);

	if (gz->level == Z_BEST_SPEED)
		extraflags |= GZ_EXTRAFLAG_FAST;
	else if (gz->level == Z_BEST_COMPRESSION)
		extraflags |= GZ_EXTRAFLAG_SLOW;

	buf[0] = GZ_MAGIC1;
	buf[1] = GZ_MAGIC2;
	buf[2] = GZ_METHOD_DEFLATE;
	buf[3] = flags;
	gzip_set32(gz->mtime, buf + 4);
	buf[8] = extraflags;
	buf[9] = gz->os_code;
	zstream_append_buffer(&gz->z, buf, sizeof(buf));

	if (!NIL_P(gz->orig_name)) {
		/* RSTRING(str)->ptr[RSTRING(str)->len] == 0. see string.c */
		zstream_append_buffer(&gz->z, RSTRING(gz->orig_name)->ptr,
				      RSTRING(gz->orig_name)->len + 1);
	}
	if (!NIL_P(gz->comment)) {
		/* ditto. */
		zstream_append_buffer(&gz->z, RSTRING(gz->comment)->ptr,
				      RSTRING(gz->comment)->len + 1);
	}

	gz->z.flag |= GZIP_FLAG_HEADER_FINISHED;
}



static VALUE gzip_read_until_zero(struct gzfile *gz, VALUE *readbuf)
{
	VALUE buf, dst = Qnil;
	char *p;
	long n;

	buf = *readbuf;
	p = memchr(RSTRING(buf)->ptr, 0, RSTRING(buf)->len);
	if (!p) {
		dst = buf;
		for (;;) {
			buf = gzip_read_io(gz, 256);
			if (NIL_P(buf))
				rb_raise(cGzError, "unexpected end of file");
			p = memchr(RSTRING(buf)->ptr, 0, RSTRING(buf)->len);
			if (p)
				break;
			rb_str_cat(dst, RSTRING(buf)->ptr, RSTRING(buf)->len);
		}
	}

	n = p - RSTRING(buf)->ptr;
	*readbuf = rb_str_substr(buf, n + 1, RSTRING(buf)->len - (n + 1));
	if (NIL_P(dst))
		dst = rb_str_resize(buf, n);
	else
		rb_str_cat(dst, RSTRING(buf)->ptr, n);
	return dst;
}



static void gzip_read_header(struct gzfile *gz)
{
	VALUE buf;
	unsigned char flags;
	long len;
	unsigned char *head;

	buf = gzip_read_io(gz, 256);
	if (NIL_P(buf) || RSTRING(buf)->len < 10)  /* the size of .gz header */
		rb_raise(cGzError, "not in gzip format");

	head = RSTRING(buf)->ptr;

	if (head[0] != GZ_MAGIC1 || head[1] != GZ_MAGIC2)
		rb_raise(cGzError, "not in gzip format");
	if (head[2] != GZ_METHOD_DEFLATE)
		rb_raise(cGzError, "unsupported compression method %d",
			 head[2]);

	flags = head[3];
	if (flags & GZ_FLAG_MULTIPART)
		rb_raise(cGzError, "multi-part gzip file is not supported");
	else if (flags & GZ_FLAG_ENCRYPT)
		rb_raise(cGzError, "encrypted gzip file is not supported");
	else if (flags & GZ_FLAG_UNKNOWN_MASK)
		rb_raise(cGzError, "unknown flags 0x%02x", flags);

	gz->mtime = gzip_get32(head + 4);
	gz->os_code = head[9];

	memmove(RSTRING(buf)->ptr, RSTRING(buf)->ptr + 10,
		RSTRING(buf)->len - 10);
	rb_str_resize(buf, RSTRING(buf)->len - 10);

	if (flags & GZ_FLAG_EXTRA) {
		if (RSTRING(buf)->len < 2)
			rb_raise(cGzError, "unexpected end of file");
		len = gzip_get16(RSTRING(buf)->ptr);

		if (RSTRING(buf)->len >= len + 2) {
			len = RSTRING(buf)->len - (len + 2);
			memmove(RSTRING(buf)->ptr,
				RSTRING(buf)->ptr + (len + 2), len);
			rb_str_resize(buf, len);
		} else {
			len = (len + 2) - RSTRING(buf)->len;
			buf = gzip_read_io(gz, len);
			if (NIL_P(buf) || RSTRING(buf)->len < len)
				rb_raise(cGzError, "unexpected end of file");
			rb_str_resize(buf, 0);
		}
	}

	if (flags & GZ_FLAG_ORIG_NAME) {
		gz->orig_name = gzip_read_until_zero(gz, &buf);
		if (NIL_P(gz->orig_name))
			rb_raise(cGzError, "unexpected end of file");
	}

	if (flags & GZ_FLAG_COMMENT) {
		gz->comment = gzip_read_until_zero(gz, &buf);
		if (NIL_P(gz->comment))
			rb_raise(cGzError, "unexpected end of file");
	}

	gzipreader_inflate(gz, buf);
}



static void gzipreader_check_footer(struct gzfile *gz)
{
	VALUE str;
	int n;
	unsigned long crc, length;
	gz->z.flag |= GZIP_FLAG_FOOTER_FINISHED;

	if (NIL_P(gz->unused))
		n = 8;
	else if (RSTRING(gz->unused)->len < 8)  /* .gz footer size */
		n = 8 - RSTRING(gz->unused)->len;
	else
		n = 0;

	if (n > 0) {
		str = gzip_read_io(gz, n);
		if (!NIL_P(str)) {
			if (NIL_P(gz->unused))
				gz->unused = str;
			else
				rb_str_cat(gz->unused, RSTRING(str)->ptr,
					   RSTRING(str)->len);
		}
		if (NIL_P(gz->unused) || RSTRING(gz->unused)->len < 8)
			rb_raise(cNoFooter, "footer is not found");
	}

	gz->z.stream.total_in += 8;  /* to rewind correctly */
	crc = gzip_get32(RSTRING(gz->unused)->ptr);
	length = gzip_get32(RSTRING(gz->unused)->ptr + 4);

	memmove(RSTRING(gz->unused)->ptr, RSTRING(gz->unused)->ptr + 8,
		RSTRING(gz->unused)->len - 8);
	rb_str_resize(gz->unused, RSTRING(gz->unused)->len - 8);

	if (gz->crc != crc)
		rb_raise(cCRCError,
			 "invalid compressed data -- crc error");
	if (gz->z.stream.total_out != length)
		rb_raise(cLengthError,
			 "invalid compressed data -- length error");
}



static void gzipreader_close(struct gzfile *gz)
{
	if (GZFILE_ZSTREAM_FINISHED(gz)
	    && !(gz->z.flag & GZIP_FLAG_FOOTER_FINISHED))
		gzipreader_check_footer(gz);
	else
		gz->unused = Qnil;

	gz->io = Qnil;
	gz->orig_name = Qnil;
	gz->comment = Qnil;
	zstream_end(&gz->z);
}



static void gzipwriter_close(struct gzfile *gz)
{
	char buf[8];  /* 8 is the size of .gz footer */

	if (!(gz->z.flag & GZIP_FLAG_HEADER_FINISHED))
		gzip_make_header(gz);
	gz->z.flag |= GZIP_FLAG_FOOTER_FINISHED;

	gzip_set32(gz->crc, buf);
	gzip_set32(gz->z.stream.total_in, buf + 4);

	zstream_run(&gz->z, Qnil, Z_FINISH);
	zstream_append_buffer(&gz->z, buf, sizeof(buf));
	gzip_write_io(gz, zstream_detach_buffer(&gz->z));

	gz->io = Qnil;
	gz->orig_name = Qnil;
	gz->comment = Qnil;
	zstream_end(&gz->z);
}



#define GZIPWRITER_INIT(gz, io) \
	GZFILE_INIT((gz), DEFLATE, (io), gzipwriter_close)
#define GZIPREADER_INIT(gz, io) \
	GZFILE_INIT((gz), INFLATE, (io), gzipreader_close)



static VALUE gzipwriter_new(VALUE klass, VALUE io,
			    VALUE v_level, VALUE v_strategy)
{
	VALUE dst;
	int level, strategy;
	int err;
	struct gzfile *gz;

	level    = value_to_compression_level(v_level);
	strategy = value_to_strategy(v_strategy);

	dst = DATA_MAKE_GZFILE(klass, gz);
	GZIPWRITER_INIT(gz, io);
	gz->level = level;

	/* this is undocumented feature of zlib */
	err = deflateInit2(&gz->z.stream, level,
			   Z_DEFLATED, -MAX_WBITS, DEF_MEM_LEVEL, strategy);
	if (err != Z_OK)
		raise_zlib_error(err, gz->z.stream.msg);
	ZSTREAM_READY(&gz->z);

	if (rb_iterator_p())
		return rb_ensure(rb_yield, dst, gzip_safe_close, dst);
	return dst;
}



VALUE rb_gzipwriter_s_new(int argc, VALUE *argv, VALUE klass)
{
	VALUE io, level, strategy;
	rb_scan_args(argc, argv, "12", &io, &level, &strategy);
	return gzipwriter_new(klass, io, level, strategy);
}



VALUE rb_gzipwriter_s_open(int argc, VALUE *argv, VALUE klass)
{
	VALUE filename, io, level, strategy;
	rb_scan_args(argc, argv, "12", &filename, &level, &strategy);
	io = rb_file_open(STR2CSTR(filename), "w");
	return gzipwriter_new(klass, io, level, strategy);
}



VALUE rb_gzipreader_s_new(VALUE klass, VALUE io)
{
	VALUE dst;
	int err;
	struct gzfile *gz;

	dst = DATA_MAKE_GZFILE(klass, gz);
	GZIPREADER_INIT(gz, io);

	/* this is undocumented feature of zlib */
	err = inflateInit2(&gz->z.stream, -MAX_WBITS);
	if (err != Z_OK)
		raise_zlib_error(err, gz->z.stream.msg);
	ZSTREAM_READY(&gz->z);

	gzip_read_header(gz);

	if (rb_iterator_p())
		return rb_ensure(rb_yield, dst, gzip_safe_close, dst);
	return dst;
}



VALUE rb_gzipreader_s_open(VALUE klass, VALUE filename)
{
	VALUE io;
	io = rb_file_open(STR2CSTR(filename), "r");
	return rb_gzipreader_s_new(klass, io);
}



VALUE rb_gzipwriter_sync(VALUE obj)
{
	return (get_gzfile(obj)->z.flag & GZIP_FLAG_SYNC) ? Qtrue : Qfalse;
}



VALUE rb_gzipwriter_set_sync(VALUE obj, VALUE mode)
{
	struct gzfile *gz = get_gzfile(obj);

	if (RTEST(mode))
		gz->z.flag |= GZIP_FLAG_SYNC;
	else
		gz->z.flag &= ~GZIP_FLAG_SYNC;
	return mode;
}



VALUE rb_gzipwriter_pos(VALUE obj)
{
	return rb_uint2inum(get_gzfile(obj)->z.stream.total_in);
}



VALUE rb_gzipwriter_write(VALUE obj, VALUE str)
{
	struct gzfile *gz = get_gzfile(obj);
	int n;

	if (TYPE(str) != T_STRING)
		str = rb_obj_as_string(str);
	n = RSTRING(str)->len;
	gz->crc = crc32(gz->crc, RSTRING(str)->ptr, n);

	if (!(gz->z.flag & GZIP_FLAG_HEADER_FINISHED))
		gzip_make_header(gz);

	if (n > 0) {
		zstream_run(&gz->z, str,
			    (gz->z.flag & GZIP_FLAG_SYNC)
			    ? Z_FULL_FLUSH : Z_NO_FLUSH);
		if (gz->z.buf_filled > 0)
			gzip_write_io(gz, zstream_detach_buffer(&gz->z));
	}

	return INT2FIX(n);
}



VALUE rb_gzipwriter_putc(VALUE obj, VALUE ch)
{
	struct gzfile *gz = get_gzfile(obj);
	char c = NUM2CHR(ch);

	if (!(gz->z.flag & GZIP_FLAG_HEADER_FINISHED))
		gzip_make_header(gz);

	gz->crc = crc32(gz->crc, &c, 1);
	gz->z.stream.next_in = &c;
	gz->z.stream.avail_in = 1;

	if (!(gz->z.flag & GZIP_FLAG_SYNC))
		zstream_run_internal(&gz->z, Z_NO_FLUSH);
	else
		zstream_run_internal(&gz->z, Z_FULL_FLUSH);

	if (gz->z.buf_filled > 0)
		gzip_write_io(gz, zstream_detach_buffer(&gz->z));

	return ch;
}



/*
 * addstr, puts, print, printf functions are borrowed from io.c.
 * I hope to remove static specifiers from definitions of these functions.
 */
VALUE rb_gzipwriter_addstr(VALUE out, VALUE str)
{
	rb_gzipwriter_write(out, str);
	return out;
}



VALUE rb_gzipwriter_printf(int argc, VALUE *argv, VALUE out)
{
	rb_gzipwriter_write(out, rb_f_sprintf(argc, argv));
	return Qnil;
}



VALUE rb_gzipwriter_print(int argc, VALUE *argv, VALUE out)
{
	int i;
	VALUE line;

	/* if no argument given, print `$_' */
	if (argc == 0) {
		argc = 1;
		line = rb_lastline_get();
		argv = &line;
	}
	for (i = 0; i < argc; i++, argv++) {
		if (!NIL_P(rb_output_fs) && i > 0)
			rb_gzipwriter_write(out, rb_output_fs);
		if (NIL_P(*argv))
			rb_gzipwriter_write(out, rb_str_new2("nil"));
		else
			rb_gzipwriter_write(out, *argv);
	}
	if (!NIL_P(rb_output_rs))
		rb_gzipwriter_write(out, rb_output_rs);
	return Qnil;
}



VALUE rb_gzipwriter_puts(int, VALUE *, VALUE);

static VALUE rb_gzipwriter_puts_ary(VALUE ary, VALUE out)
{
	VALUE tmp;
	int i;

	for (i = 0; i < RARRAY(ary)->len; i++) {
		tmp = RARRAY(ary)->ptr[i];
		if (rb_inspecting_p(tmp))
			tmp = rb_str_new2("[...]");
		rb_gzipwriter_puts(1, &tmp, out);
	}
	return Qnil;
}


VALUE rb_gzipwriter_puts(int argc, VALUE *argv, VALUE out)
{
	int i;
	VALUE line;

	/* if no argument given, print newline. */
	if (argc == 0) {
		rb_io_write(out, rb_default_rs);
		return Qnil;
	}

	for (i = 0; i < argc; i++, argv++) {
		switch (TYPE(*argv)) {
		  case T_NIL:
			line = rb_str_new2("nil");
			break;
		  case T_ARRAY:
			rb_protect_inspect(rb_gzipwriter_puts_ary, *argv, out);
			continue;
		  default:
			line = argv[i];
			break;
		}
		line = rb_obj_as_string(line);
		rb_gzipwriter_write(out, line);
		if (RSTRING(line)->ptr[RSTRING(line)->len - 1] != '\n')
			rb_gzipwriter_write(out, rb_default_rs);
        }
	return Qnil;
}



static void gzipreader_read_data(struct gzfile *gz)
{
	while (!(gz->z.flag & ZSTREAM_FLAG_FINISHED)) {
		gzipreader_inflate(gz, gzip_read_io(gz, GZIP_READ_SIZE));
		if (gz->z.buf_filled > 0)
			break;
	}
}



VALUE rb_gzipreader_pos(VALUE obj)
{
	struct gzfile *gz = get_gzfile(obj);
	return rb_uint2inum(gz->z.stream.total_out - gz->z.buf_filled);
}



VALUE rb_gzipreader_rewind(VALUE obj)
{
	struct gzfile *gz = get_gzfile(obj);
	long n;

	n = -(gz->z.stream.total_in);
	if (!NIL_P(gz->unused))
		n -= RSTRING(gz->unused)->len;

	rb_funcall(gz->io, id_seek, 2, rb_int2inum(n), INT2FIX(1));
	zstream_reset(&gz->z);
	gz->crc = crc32(0, Z_NULL, 0);
	gz->unused = Qnil;
	/* is it OK? adapting to IO#rewind for the time being... */
	/* gz->lineno = 0; */
	gz->ungetc = 0;

	return INT2FIX(0);
}



VALUE rb_gzip_unused(VALUE obj)
{
	struct gzfile *gz;
	Data_Get_Struct(obj, struct gzfile, gz);

	if (NIL_P(gz->io))
		return gz->unused;

	if (GZFILE_ZSTREAM_FINISHED(gz)) {
		if (!(gz->z.flag & GZIP_FLAG_FOOTER_FINISHED))
			gzipreader_check_footer(gz);
		return gz->unused;
	}
	return Qnil;
}



static VALUE gzipreader_result(struct gzfile *gz, VALUE dst)
{
	if (RSTRING(dst)->len <= gz->ungetc)
		gz->ungetc -= RSTRING(dst)->len;
	else {
		gz->crc = crc32(gz->crc, RSTRING(dst)->ptr + gz->ungetc,
				RSTRING(dst)->len - gz->ungetc);
		gz->ungetc = 0;
	}
	OBJ_TAINT(dst);
	return dst;
}



static VALUE gzipreader_read_all(struct gzfile *gz)
{
	while (!(gz->z.flag & ZSTREAM_FLAG_FINISHED))
		gzipreader_read_data(gz);

	if (gz->z.buf_filled == 0) {
		if (!(gz->z.flag & GZIP_FLAG_FOOTER_FINISHED))
			gzipreader_check_footer(gz);
		return Qnil;
	}
	return gzipreader_result(gz, zstream_detach_buffer(&gz->z));
}



VALUE rb_gzipreader_read(int argc, VALUE *argv, VALUE obj)
{
	struct gzfile *gz = get_gzfile(obj);
	VALUE vlen;
	int len;

	rb_scan_args(argc, argv, "01", &vlen);
	if (NIL_P(vlen))
		return gzipreader_read_all(gz);

	len = NUM2INT(vlen);
	if (len < 0)
		rb_raise(rb_eArgError, "negative length %d given", len);

	while (!(gz->z.flag & ZSTREAM_FLAG_FINISHED)) {
		gzipreader_read_data(gz);
		if (gz->z.buf_filled >= len)
			break;
	}

	if (gz->z.buf_filled == 0) {
		if (!(gz->z.flag & GZIP_FLAG_FOOTER_FINISHED))
			gzipreader_check_footer(gz);
		return Qnil;
	}
	return gzipreader_result(gz, zstream_shift_buffer(&gz->z, len));
}



VALUE rb_gzipreader_getc(VALUE obj)
{
	struct gzfile *gz = get_gzfile(obj);
	VALUE dst;
	unsigned char c;

	if (gz->z.buf_filled == 0) {
		gzipreader_read_data(gz);
		if (gz->z.buf_filled == 0) {
			if (!(gz->z.flag & GZIP_FLAG_FOOTER_FINISHED))
				gzipreader_check_footer(gz);
			return Qnil;
		}
	}

	dst = zstream_shift_buffer(&gz->z, 1);
	c = RSTRING(dst)->ptr[0];
	if (gz->ungetc > 0)
		gz->ungetc--;
	else
		gz->crc = crc32(gz->crc, &c, 1);

	return INT2FIX(c);
}



VALUE rb_gzipreader_readchar(VALUE obj)
{
	VALUE dst;
	dst = rb_gzipreader_getc(obj);
	if (NIL_P(dst))
		rb_raise(rb_eEOFError, "End of file reached");
	return dst;
}



VALUE rb_gzipreader_each_byte(VALUE obj)
{
	VALUE c;
	while (!NIL_P(c = rb_gzipreader_getc(obj)))
		rb_yield(c);
	return Qnil;
}



VALUE rb_gzipreader_ungetc(VALUE obj, VALUE ch)
{
	struct gzfile *gz = get_gzfile(obj);
	zstream_buffer_ungetc(&gz->z, NUM2CHR(ch));
	gz->ungetc++;
	return Qnil;
}



static void gzipreader_skip_linebreak(struct gzfile *gz)
{
	VALUE str;
	char *p;
	int n;

	n = 0;
	if (gz->z.buf_filled > 0)
		p = RSTRING(gz->z.buf)->ptr;

	do {
		if (n >= gz->z.buf_filled) {
			if (gz->z.buf_filled > 0) {
				str = zstream_detach_buffer(&gz->z);
				gz->crc = crc32(gz->crc, RSTRING(str)->ptr,
						RSTRING(str)->len);
			}
			gzipreader_read_data(gz);
			if (gz->z.buf_filled == 0)
				return;
			n = 0;
			p = RSTRING(gz->z.buf)->ptr;
		}
	} while (n++, *(p++) == '\n');

	str = zstream_shift_buffer(&gz->z, n - 1);
	gz->crc = crc32(gz->crc, RSTRING(str)->ptr, RSTRING(str)->len);
}



static VALUE gzipreader_gets(int argc, VALUE *argv, VALUE obj)
{
	struct gzfile *gz = get_gzfile(obj);
	VALUE rs, dst;
	char *rsptr, *p;
	long rslen, n;
	int rspara;

	if (argc == 0)
		rs = rb_rs;
	else {
		rb_scan_args(argc, argv, "1", &rs);
		if (!NIL_P(rs))
			Check_Type(rs, T_STRING);
	}

	if (NIL_P(rs)) {
		dst = gzipreader_read_all(gz);
		if (!NIL_P(dst))
			gz->lineno++;
		return dst;
	}

	if (RSTRING(rs)->len == 0) {
		rsptr = "\n\n";
		rslen = 2;
		rspara = 1;
	} else {
		rsptr = RSTRING(rs)->ptr;
		rslen = RSTRING(rs)->len;
		rspara = 0;
	}

	if (rspara)
		gzipreader_skip_linebreak(gz);

	if (gz->z.buf_filled == 0) {
		gzipreader_read_data(gz);
		if (gz->z.buf_filled == 0) {
			if (!(gz->z.flag & GZIP_FLAG_FOOTER_FINISHED))
				gzipreader_check_footer(gz);
			return Qnil;
		}
	}

	n = rslen;
	p = RSTRING(gz->z.buf)->ptr;
	for (;;) {
		if (n > gz->z.buf_filled) {
			if (gz->z.flag & ZSTREAM_FLAG_FINISHED)
				break;
			gzipreader_read_data(gz);
			p = RSTRING(gz->z.buf)->ptr + n - rslen;
		}
		if (memcmp(p, rsptr, rslen) == 0)
			break;
		p++, n++;
	}

	gz->lineno++;
	dst = gzipreader_result(gz, zstream_shift_buffer(&gz->z, n));

	if (rspara)
		gzipreader_skip_linebreak(gz);
	return dst;
}



VALUE rb_gzipreader_gets(int argc, VALUE *argv, VALUE obj)
{
	VALUE dst;
	dst = gzipreader_gets(argc, argv, obj);
	if (!NIL_P(dst))
		rb_lastline_set(dst);
	return dst;
}



VALUE rb_gzipreader_readline(int argc, VALUE *argv, VALUE obj)
{
	VALUE dst;
	dst = rb_gzipreader_gets(argc, argv, obj);
	if (NIL_P(dst))
		rb_raise(rb_eEOFError, "End of file reached");
	return dst;
}



VALUE rb_gzipreader_each(int argc, VALUE *argv, VALUE obj)
{
	VALUE str;
	while (!NIL_P(str = gzipreader_gets(argc, argv, obj)))
		rb_yield(str);
	return obj;
}



VALUE rb_gzipreader_readlines(int argc, VALUE *argv, VALUE obj)
{
	VALUE str, dst;
	dst = rb_ary_new();
	while (!NIL_P(str = gzipreader_gets(argc, argv, obj)))
		rb_ary_push(dst, str);
	return dst;
}

#endif /* GZIP_SUPPORT */




void Init_zlib()
{
	VALUE mZlib, cZStream, cDeflate, cInflate;
#if GZIP_SUPPORT
	VALUE cGzip, cGzipWriter, cGzipReader;
#endif

	mZlib = rb_define_module("Zlib");

	cZError = rb_define_class_under(mZlib, "Error", rb_eStandardError);
	cStreamEnd    = rb_define_class_under(mZlib, "StreamEnd", cZError);
	cNeedDict     = rb_define_class_under(mZlib, "NeedDict", cZError);
	cDataError    = rb_define_class_under(mZlib, "DataError", cZError);
	cStreamError  = rb_define_class_under(mZlib, "StreamError", cZError);
	cMemError     = rb_define_class_under(mZlib, "MemError", cZError);
	cBufError     = rb_define_class_under(mZlib, "BufError", cZError);
	cVersionError = rb_define_class_under(mZlib, "VersionError", cZError);

	rb_define_module_function(mZlib, "version", rb_zlib_version, 0);
#if 0
	rb_define_module_function(mZlib, "compress", rb_zlib_compress, -1);
	rb_define_module_function(mZlib, "uncompress", rb_zlib_uncompress, -1);
#endif
	rb_define_module_function(mZlib, "adler32", rb_zlib_adler32, -1);
	rb_define_module_function(mZlib, "crc32", rb_zlib_crc32, -1);
	rb_define_module_function(mZlib, "crc_table", rb_zlib_crc_table, 0);

	cZStream = rb_define_class_under(mZlib, "ZStream", rb_cObject);

	rb_undef_method(CLASS_OF(cZStream), "new");
	rb_define_method(cZStream, "total_in", rb_zstream_total_in, 0);
	rb_define_method(cZStream, "total_out", rb_zstream_total_out, 0);
	rb_define_method(cZStream, "data_type", rb_zstream_data_type, 0);
	rb_define_method(cZStream, "adler", rb_zstream_adler, 0);
	rb_define_method(cZStream, "finished?", rb_zstream_finished_p, 0);
	rb_define_method(cZStream, "closed?", rb_zstream_closed_p, 0);
	rb_define_method(cZStream, "close", rb_zstream_end, 0);
	rb_define_method(cZStream, "end", rb_zstream_end, 0);
	rb_define_method(cZStream, "reset", rb_zstream_reset, 0);
	rb_define_method(cZStream, "finish", rb_zstream_finish, 0);
	rb_define_method(cZStream, "flush_out", rb_zstream_flush_out, 0);

	rb_define_const(cZStream, "BINARY", INT2FIX(Z_BINARY));
	rb_define_const(cZStream, "ASCII", INT2FIX(Z_ASCII));
	rb_define_const(cZStream, "UNKNOWN", INT2FIX(Z_UNKNOWN));


	cDeflate = rb_define_class("Deflate", cZStream);

	rb_define_singleton_method(cDeflate, "deflate",
				   rb_deflate_s_deflate, -1);

	rb_define_singleton_method(cDeflate, "new", rb_deflate_s_new, -1);
	rb_define_method(cDeflate, "clone", rb_deflate_clone, 0);
	rb_define_method(cDeflate, "deflate", rb_deflate_deflate, -1);
	rb_define_method(cDeflate, "<<", rb_deflate_addstr, 1);
	rb_define_method(cDeflate, "params", rb_deflate_params, 2);
	rb_define_method(cDeflate, "set_dictionary",
			 rb_deflate_set_dictionary, 1);

	rb_define_const(cDeflate, "NO_COMPRESSION", INT2FIX(Z_NO_COMPRESSION));
	rb_define_const(cDeflate, "BEST_SPEED", INT2FIX(Z_BEST_SPEED));
	rb_define_const(cDeflate, "BEST_COMPRESSION",
			INT2FIX(Z_BEST_COMPRESSION));
	rb_define_const(cDeflate, "DEFAULT_COMPRESSION",
			INT2FIX(Z_DEFAULT_COMPRESSION));

	rb_define_const(cDeflate, "FILTERED", INT2FIX(Z_FILTERED));
	rb_define_const(cDeflate, "HUFFMAN_ONLY", INT2FIX(Z_HUFFMAN_ONLY));
	rb_define_const(cDeflate, "DEFAULT_STRATEGY",
			INT2FIX(Z_DEFAULT_STRATEGY));

	rb_define_const(cDeflate, "MAX_WBITS", INT2FIX(MAX_WBITS));
	rb_define_const(cDeflate, "DEF_MEM_LEVEL", INT2FIX(DEF_MEM_LEVEL));
	rb_define_const(cDeflate, "MAX_MEM_LEVEL", INT2FIX(MAX_MEM_LEVEL));

	rb_define_const(cDeflate, "NO_FLUSH", INT2FIX(Z_NO_FLUSH));
	rb_define_const(cDeflate, "SYNC_FLUSH", INT2FIX(Z_SYNC_FLUSH));
	rb_define_const(cDeflate, "FULL_FLUSH", INT2FIX(Z_FULL_FLUSH));


	cInflate = rb_define_class("Inflate", cZStream);

	rb_define_singleton_method(cInflate, "inflate",
				   rb_inflate_s_inflate, 1);

	rb_define_singleton_method(cInflate, "new", rb_inflate_s_new, -1);
	rb_define_method(cInflate, "inflate", rb_inflate_inflate, 1);
	rb_define_method(cInflate, "<<", rb_inflate_addstr, 1);
#if 0
	rb_define_method(cInflate, "sync", rb_inflate_sync, 0);
#endif
	rb_define_method(cInflate, "sync_point?", rb_inflate_sync_point_p, 0);
	rb_define_method(cInflate, "set_dictionary",
			 rb_inflate_set_dictionary, 1);

	rb_define_const(cInflate, "MAX_WBITS", INT2FIX(MAX_WBITS));

#if GZIP_SUPPORT
	id_write = rb_intern("write");
	id_read = rb_intern("read");
	id_flush = rb_intern("flush");
	id_seek = rb_intern("seek");
	id_close = rb_intern("close");

	cGzip = rb_define_class_under(mZlib, "Gzip", rb_cObject);
	cGzError = rb_define_class_under(cGzip, "Error", cZError);

	cGzipWriter = rb_define_class("GzipWriter", cGzip);
	cGzipReader = rb_define_class("GzipReader", cGzip);

	rb_define_method(cGzip, "to_io", rb_gzip_to_io, 0);
	rb_define_method(cGzip, "crc", rb_gzip_crc, 0);
	rb_define_method(cGzip, "mtime", rb_gzip_mtime, 0);
	rb_define_method(cGzipWriter, "level", rb_gzip_level, 0);
	rb_define_method(cGzip, "os_code", rb_gzip_os_code, 0);
	rb_define_method(cGzip, "orig_name", rb_gzip_orig_name, 0);
	rb_define_method(cGzip, "comment", rb_gzip_comment, 0);
	rb_define_method(cGzipReader, "unused", rb_gzip_unused, 0);
	rb_define_method(cGzipReader, "lineno", rb_gzip_lineno, 0);
	rb_define_method(cGzipReader, "lineno=", rb_gzip_set_lineno, 1);
	rb_define_method(cGzipWriter, "mtime=", rb_gzip_set_mtime, 1);
	rb_define_method(cGzipWriter, "orig_name=", rb_gzip_set_orig_name, 1);
	rb_define_method(cGzipWriter, "comment=", rb_gzip_set_comment, 1);
	rb_define_method(cGzip, "close", rb_gzip_close, -1);
	rb_define_method(cGzip, "closed?", rb_gzip_closed_p, 0);
	rb_define_method(cGzipReader, "eof", rb_gzip_eof_p, 0);
	rb_define_method(cGzipReader, "eof?", rb_gzip_eof_p, 0);

	rb_define_singleton_method(cGzipWriter, "new",
				   rb_gzipwriter_s_new, -1);
	rb_define_singleton_method(cGzipWriter, "open",
				   rb_gzipwriter_s_open, -1);
	rb_define_singleton_method(cGzipReader, "new",
				   rb_gzipreader_s_new, 1);
	rb_define_singleton_method(cGzipReader, "open",
				   rb_gzipreader_s_open, 1);

	rb_define_method(cGzipWriter, "sync", rb_gzipwriter_sync, 0);
	rb_define_method(cGzipWriter, "sync=", rb_gzipwriter_set_sync, 1);
	rb_define_method(cGzipWriter, "pos", rb_gzipwriter_pos, 0);
	rb_define_method(cGzipWriter, "tell", rb_gzipwriter_pos, 0);
	rb_define_method(cGzipWriter, "write", rb_gzipwriter_write, 1);
	rb_define_method(cGzipWriter, "putc", rb_gzipwriter_putc, 1);
	rb_define_method(cGzipWriter, "<<", rb_gzipwriter_addstr, 1);
	rb_define_method(cGzipWriter, "puts", rb_gzipwriter_puts, -1);
	rb_define_method(cGzipWriter, "print", rb_gzipwriter_print, -1);
	rb_define_method(cGzipWriter, "printf", rb_gzipwriter_printf, -1);

	cNoFooter = rb_define_class_under(cGzipReader, "NoFooter", cGzError);
	cCRCError = rb_define_class_under(cGzipReader, "CRCError", cGzError);
	cLengthError = rb_define_class_under(cGzipReader, "LengthError",
					     cGzError);

	rb_define_method(cGzipReader, "pos", rb_gzipreader_pos, 0);
	rb_define_method(cGzipReader, "tell", rb_gzipreader_pos, 0);
	rb_define_method(cGzipReader, "rewind", rb_gzipreader_rewind, 0);
	rb_define_method(cGzipReader, "read", rb_gzipreader_read, -1);
	rb_define_method(cGzipReader, "getc", rb_gzipreader_getc, 0);
	rb_define_method(cGzipReader, "readchar", rb_gzipreader_readchar, 0);
	rb_define_method(cGzipReader, "each_byte", rb_gzipreader_each_byte, 0);
	rb_define_method(cGzipReader, "ungetc", rb_gzipreader_ungetc, 1);
	rb_define_method(cGzipReader, "gets", rb_gzipreader_gets, -1);
	rb_define_method(cGzipReader, "readline", rb_gzipreader_readline, -1);
	rb_define_method(cGzipReader, "each", rb_gzipreader_each, -1);
	rb_define_method(cGzipReader, "each_line", rb_gzipreader_each, -1);
	rb_define_method(cGzipReader, "readlines", rb_gzipreader_readlines,-1);

	rb_include_module(cGzipReader, rb_mEnumerable);

	rb_define_const(cGzipReader, "OS_CODE", INT2FIX(OS_CODE));
	rb_define_const(cGzipReader, "OS_MSDOS", INT2FIX(OS_MSDOS));
	rb_define_const(cGzipReader, "OS_AMIGA", INT2FIX(OS_AMIGA));
	rb_define_const(cGzipReader, "OS_VMS", INT2FIX(OS_VMS));
	rb_define_const(cGzipReader, "OS_UNIX", INT2FIX(OS_UNIX));
	rb_define_const(cGzipReader, "OS_ATARI", INT2FIX(OS_ATARI));
	rb_define_const(cGzipReader, "OS_OS2", INT2FIX(OS_OS2));
	rb_define_const(cGzipReader, "OS_MACOS", INT2FIX(OS_MACOS));
	rb_define_const(cGzipReader, "OS_TOPS20", INT2FIX(OS_TOPS20));
	rb_define_const(cGzipReader, "OS_WIN32", INT2FIX(OS_WIN32));
#endif /* GZIP_SUPPORT */
}
